<?php

namespace App\Controller\Entreprises;

use App\Entity\PaiementEntreprise;
use App\Entity\PaniersEntreprise;
use App\Entity\Proformaentreprises;
use App\Entity\Searchs\SearchEntrepriesProforma;
use App\Form\Entreprises\PaniersEntrepriseType;
use App\Form\Entreprises\ProformaTitreDateType;
use App\Form\Entreprises\ProformaType;
use App\Form\Entreprises\Searchs\SearchEntreprisesProformaType;
use App\Managers\ManagersInterface;
use App\Repository\ClientsentreprisesRepository;
use App\Repository\PaniersEntrepriseRepository;
use App\Repository\ProformaentreprisesRepository;
use App\Services\Helpers\FlashServiceInterface;
use App\Services\Helpers\HelpersInterfaces;
use DateTime;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/dashboard/entreprise/proformas')]
#[IsGranted(new Expression('is_granted("ROLE_SUPER_ADMIN") or is_granted("ROLE_ENTREPRISE")'))]
class ProformaentreprisesController extends AbstractController
{
    public function __construct(
        private readonly EntityManagerInterface        $entityManager,
        private readonly ClientsentreprisesRepository  $clientsentreprisesRepository,
        private readonly ProformaentreprisesRepository $proformaentreprisesRepository,
        private readonly PaniersEntrepriseRepository   $paniersEntrepriseRepository,
        private readonly ManagersInterface             $manager,
        private readonly FlashServiceInterface         $flash,
        private readonly RequestStack                  $requestStack
    )
    {
    }

    /**
     * @throws Exception
     */
    #[Route('/', name: "proformaentreprises.index", methods: ['GET', 'POST'])]
    public function index(Request $request, ProformaentreprisesRepository $proformaentreprisesRepository): Response
    {
        $search = new SearchEntrepriesProforma();
        $form = $this->createForm(SearchEntreprisesProformaType::class, $search, ['search' => $search]);
        $form->handleRequest($request);
        $search->page = $request->query->getInt('page', 1);
        //$search->user = $this->getUser();
        $proformas = $proformaentreprisesRepository->findProformaByPeriode($search);
        if ($form->isSubmitted() && $form->isValid()) {
            $proformas = $proformaentreprisesRepository->findProformaByPeriode($search);
        }

        return $this->render('entreprises/proformas/index.html.twig', [
            'search' => $search,
            'proformaentreprises' => $proformas,
            'title' => 'Liste des factures proformas',
            'libelle_ajouter' => 'Nouvelle proforma',
            'active' => 'proformaentreprises.index',
            'form' => $form->createView(),
        ]);
    }

    /**
     * @throws Exception
     */
    #[Route('/ajouter-bon-de-commande/{id}', name: 'proformaentreprises.ajouter.numero.bon.de.commande', requirements: ['id' => Requirement::DIGITS], methods: ['GET', 'POST'])]
    public function ajouterBonDeCommande(Proformaentreprises $proformaEntreprise): Response
    {
        if (isset($_POST['envoyer'])) {
            $numeroBon = $_POST['numeroBon'];
            $dateBon = $_POST['dateBon'];

            $dateBon = preg_replace("#(/)#", '-', $dateBon);

            $dateBon = new DateTime($dateBon);

            $proformaEntreprise->setNumeroBonDeCommande($numeroBon);
            $proformaEntreprise->setDateBonDeCommande($dateBon);
            $this->manager->setEntity($proformaEntreprise, 'edit');
            $this->flash->success("Modification effectuée avec succès.");
        }

        return $this->render('entreprises/proformas/ajouter-numero-bon-de-commande.html.twig', [
            'proformaentreprises' => $proformaEntreprise,
            'title' => 'Liste des factures proformas',
            'libelle_ajouter' => 'Nouvelle proforma'
        ]);
    }

    #[Route('/nouveau-paiement', name: "proformaentreprises.nouveau.versement", methods: ['GET'])]
    public function nouveauVersement(): Response
    {
        return $this->render('entreprises/proformas/nouveau_versement.html.twig', [
            'title' => 'Enregistrer un paiement',
            'libelle_ajouter' => 'Nouvelle proforma',
            'active' => 'proformaentreprises.index',
        ]);
    }

    #[Route('/recapitulatif-paiement', name: "proformaentreprises.recapitulatif.versement", methods: ['GET'])]
    public function recapitulatifVersement(ProformaentreprisesRepository $proformaentreprisesRepository): Response
    {
        $montantPaiement = (int)$_POST['montantPaiement'];
        $numeroFactures = $_POST['numeroFactures'];
        $numeroCheque = $_POST['numeroCheque'];
        $datePaiement = $_POST['datePaiement'];

        $numeroFactures = explode(',', $numeroFactures);

        $proformas = array();

        foreach ($numeroFactures as $numeroFacture) {
            $proforma = $proformaentreprisesRepository->findOneByNumerolabelle($numeroFacture);
            if (null != $proforma) {
                $proformas[] = array(
                    'numeroFacture' => $numeroFacture,
                    'id' => $proforma->getId(),
                    'montant' => $proforma->getTotalttc(),
                    'laDate' => $proforma->getDatefacture(),
                    'trouve' => true,
                    'client' => $proforma->getClientsentreprises()->getNom()
                );
            } else {
                $proformas[] = array(
                    'numeroFacture' => $numeroFacture,
                    'trouve' => false,
                );
            }
        }

        return $this->render('entreprises/proformas/recapitulatif_paiement.html.twig', [
            'proformas' => $proformas
        ]);
    }

    /**
     * @throws Exception
     */
    #[Route('/enregistrer-paiement', name: "proformaentreprises.enregistrer.versement", methods: ['GET', 'POST'])]
    public function enregistrerVersement(): Response
    {
        $montantPaiement = (int)$_POST['montantPaiement'];
        $numeroFactures = $_POST['numeroFactures'];
        $numeroCheque = $_POST['numeroCheque'];
        $datePaiement = $_POST['datePaiement'];

        $paiement = new PaiementEntreprise;
        $datePaiement = preg_replace("#(/)#", '-', $datePaiement);
        $datePaiement = new DateTimeImmutable($datePaiement);

        $paiement->setMontant($montantPaiement);
        $paiement->setLaDate($datePaiement);
        $paiement->setNumeroCheque($numeroCheque);
        $paiement->setNumeroFacture($numeroFactures);

        $paiement->setUser($this->getUser());

        $this->manager->setEntity($paiement, 'new');
        $this->flash->success("Paiement effectuée avec succès.");
        return $this->render('entreprises/proformas/enregistrer_paiement.html.twig');
    }

    /**
     * @throws Exception
     */
    #[Route('/nouveau-paiment/{id}', name: 'proformaentreprises.nouveau.paiement', requirements: ['id' => Requirement::DIGITS], methods: ['GET', 'POST'])]
    public function nouveauPaiement(Proformaentreprises $proformaentreprises): Response
    {
        $montantPaiement = 0;
        $detailPaiement = '';
        if (isset($_POST['envoyer'])) {
            $montantPaiement = (int)$_POST['montantPaiement'];
            $detailPaiement = $_POST['detailPaiement'];
            $datePaiement = $_POST['datePaiement'];
            $datePaiement = preg_replace("#(/)#", '-', $datePaiement);
            $datePaiement = new DateTimeImmutable($datePaiement);

            if ($montantPaiement < 0 || $proformaentreprises->getResteAPayer() < $montantPaiement) {
                $this->flash->danger("Montant non valide.");
                return $this->redirectToRoute('proformaentreprises.nouveau.paiement', array('id' => $proformaentreprises->getId()));

            }

            $paiement = new PaiementEntreprise;
            $paiement->setProformaentreprise($proformaentreprises);
            $paiement->setUser($this->getUser());
            $paiement->setMontant($montantPaiement);
            $paiement->setDetail($detailPaiement);
            $paiement->setLaDate($datePaiement);
            $proformaentreprises->setMontantPaye($proformaentreprises->getMontantPaye() + $montantPaiement);
            $proformaentreprises->setResteAPayer($proformaentreprises->getResteAPayer() - $montantPaiement);
            $this->flash->success("Paiement effectué.");
            $this->manager->setEntity($paiement, 'new');
        }

        return $this->render('entreprises/proformas/nouveau_paiement.html.twig', [
            'proformaentreprises' => $proformaentreprises,
            'title' => 'Nouveau paiement',
            'libelle_ajouter' => 'Nouvelle prestation',
            'active' => 'proformaparticuliers.index',
        ]);
    }

    #[Route('/detail-sur-facture/{id}', name: 'detail.sur.facture', requirements: ['id' => Requirement::DIGITS], methods: ['GET', 'POST'])]
    public function detailSurFacture(Proformaentreprises $proformaentreprises): Response
    {
        $observationSurFacture = '';
        if (isset($_POST['envoyer'])) {
            $observationSurFacture = $_POST['observationSurFacture'];
            $proformaentreprises->setObservationSurFacture($observationSurFacture);
            $this->manager->setEntity($proformaentreprises, 'edit');
            $this->flash->success("Enregistrement effectué avec succès.");
        }
        return $this->render('entreprises/proformas/detail_sur_facture.html.twig', [
            'title' => 'Détail sur facture',
            'libelle_ajouter' => 'Détail sur facture',
            'active' => 'proformaentreprises.index',
            'proformaentreprises' => $proformaentreprises
        ]);
    }

    #[Route('/factures/liste', name: "proformaentreprises.facture", methods: ['GET'])]
    public function factures(ProformaentreprisesRepository $proformaentreprisesRepository): Response
    {
        return $this->render('entreprises/proformas/factures.html.twig', [
            'factures' => $proformaentreprisesRepository->findFactures(),
            'title' => 'Liste des factures',
            'libelle_ajouter' => 'Nouvelle proforma',
            'active' => 'proformaentreprises.index',
        ]);
    }

    #[Route('/show/{id}', name: 'proformaentreprises.show', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function show(Proformaentreprises $proformaentreprises): Response
    {
        $roleUser = $this->isGranted('ROLE_SUPER_ADMIN');
        if ($roleUser && $proformaentreprises->getStatut() === false) {
            $proformaentreprises->setStatut(true);
            $this->manager->setEntity($proformaentreprises, 'edit');
        }

        return $this->render('entreprises/proformas/show.html.twig', [
            'proformaentreprise' => $proformaentreprises,
            'title' => 'Facture proforma',
            'active' => 'proformaentreprises.show',
        ]);
    }

    /**
     * @throws Exception
     */
    #[Route('/{id}/edit', name: 'proformaentreprises.edit', requirements: ['id' => Requirement::DIGITS], methods: ['GET', 'POST'])]
    public function edit(
        HelpersInterfaces           $helper,
        Request                     $request,
        Proformaentreprises         $proformaentreprises,
        PaniersEntrepriseRepository $paniersRepository
    ): Response
    {
        // Récupération de la session
        $session = $this->requestStack->getSession();

        // Récupération et stockage de la TVA en session
        if (!$session->has('tva')) {
            $tva = $proformaentreprises->getTva() > 0 ? $proformaentreprises->getTva() : 0;
            $session->set('tva', $tva);
        } else {
            $tva = $session->get('tva');
        }

        // Récupération et stockage de la remise en session
        if (!$session->has('remise')) {
            $remise = $proformaentreprises->getRemise() > 0 ? $proformaentreprises->getRemise() : 0;
            $session->set('remise', $remise);
        } else {
            $remise = $session->get('remise');
        }

        $products = $proformaentreprises->getProduits()['facture'];
        $dateDeLaFacture = $proformaentreprises->getDateFacture();
        if (count($paniersRepository->findAllForProforma($proformaentreprises)) <= 0 && count($products) > 0) {
            foreach ($products as $product) {
                $panier = new PaniersEntreprise();
                $panier->setDesignation($product['designation']);
                $panier->setQuantite($product['quantite']);
                $panier->setPu($product['pu']);
                $panier->setPt($product['pt']);
                $panier->setUsers($this->getUser());
                $panier->setProformaid($proformaentreprises->getId());
                $this->manager->setEntity($panier, 'new');
            }
        }
        $form_proforma = $this->createForm(ProformaType::class, $proformaentreprises);
        $form_proforma->handleRequest($request);
        if ($form_proforma->isSubmitted() && $form_proforma->isValid()) {
            $dateconvert = date("Y-m-d H:i:s", strtotime($form_proforma->get('datefacture')->getData()));
            $date = new DateTime($dateconvert);
            $proformaentreprises->setUsers($this->getUser());
            $this->manager->setEntity($proformaentreprises, 'edit');
            $this->cleanPanierByUser();
            $this->flash->success("Proforma modifiée avec succès.");
            return $this->redirectToRoute('proformaentreprises.edit', ['id' => $proformaentreprises->getId()]);
        }

        $panier = new PaniersEntreprise();
        $form = $this->createForm(PaniersEntrepriseType::class, $panier);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $pu = intval($form->get('pu')->getData() * $form->get('quantite')->getData());
            $panier->setPt($pu);
            $panier->setUsers($this->getUser());
            $panier->setProformaid($proformaentreprises->getId());
            $this->manager->setEntity($panier, 'new');
            $this->flash->success("Produit ajouté avec succès.");
            return $this->redirectToRoute('proformaentreprises.edit', ['id' => $proformaentreprises->getId()]);
        }

        $tt = $paniersRepository->findSumForProforma($proformaentreprises);
        $tt = $tt['somme'];
        $laTva = (float)$tva;
        $laRemise = (float)$remise;
        $montantTva = ($tt * $laTva / 100);
        $montantRemise = ($tt * $laRemise / 100);
        $TTCFINAL = $tt + $montantTva - $montantRemise;
        $montantlettre = strtoupper($helper->towords($TTCFINAL));

        return $this->render('entreprises/proformas/edit.html.twig', [
            'proformaentreprise' => $proformaentreprises,
            'paniers' => $paniersRepository->findAllForProforma($proformaentreprises),
            'totalHT' => $tt,
            'TTCFINAL' => $TTCFINAL,
            'form_proforma' => $form_proforma->createView(),
            'form' => $form->createView(),
            'tva' => $tva,
            'remise' => $remise,
            'montantRemise' => $montantRemise,
            'montantTva' => $montantTva,
            'panier' => $panier,
            'title' => 'Editer une proforma',
            'libelle_ajouter' => 'Nouvelle proforma',
            'libelle_liste' => 'Liste des proformas',
            'dateDeLaFacture' => $dateDeLaFacture
        ]);
    }

    private function cleanPanierByUser(): void
    {
        $paniers = $this->paniersEntrepriseRepository->findBy(['users' => $this->getUser()]);
        $session = $this->requestStack->getSession();
        foreach ($paniers as $panier) {
            $this->manager->setEntity($panier, 'delete');
        }
        $session->remove('client');
        $session->remove('tva');
        $session->remove('remise');
    }

    #[Route('/products/edit/{id}', name: 'proformaentreprises.editproducts', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function editProducts(Proformaentreprises $proformaentreprises, PaniersEntrepriseRepository $paniersRepository): Response
    {
        //On reccupère la tva en session
        $session = $this->requestStack->getSession();
        if (!$session->has('tva')) $session->set('tva', []);
        $tva = $session->get('tva');

        //On crée la session de la facture
        if (!$session->has('proforma')) $session->set('proforma', []);
        $proforma = $session->get('proforma');

        $paniers = $paniersRepository->findAll();

        if (!$paniers) {
            $this->flash->danger("Aucun produit dans le panier.");
            return $this->redirectToRoute('proformaentreprises.edit', ['id' => $proformaentreprises->getId()]);
        }

        $totalHT = 0;
        $totalTTC = 0;

        foreach ($paniers as $panier) {
            $totalHT += $panier->getPt();

            $proforma['facture'][$panier->getId()] = [
                'id' => $panier->getId(),
                'quantite' => $panier->getQuantite(),
                'designation' => $panier->getDesignation(),
                'pu' => round($panier->getPu()),
                'pt' => round($panier->getPu() * $panier->getQuantite())
            ];
        }

        $tva = (float)$tva;
        $montantTva = ($totalHT * $tva) / 100;
        $totalTTC = ceil($totalTTC + ($totalHT + $montantTva));

        $proformaentreprises->setTotalht($totalHT);
        $proformaentreprises->setTotalttc($totalTTC);
        $proformaentreprises->setTva($tva);
        $proformaentreprises->setUsers($this->getUser());
        $proformaentreprises->setProduits($proforma);
        $this->manager->setEntity($proformaentreprises, 'edit');
        $this->flash->success("Produit(s) mis à jour avec succès.");
        $this->cleanPanierByUser();
        return $this->redirectToRoute('proformaentreprises.index');
    }

    #[Route('/edit-client', name: "proformaentreprises.otherClient", methods: ['GET'])]
    public function otherClient(): Response
    {
        $session = $this->requestStack->getSession();
        if (!$session->has('client')) $session->set('client', []);
        $session->remove('client');
        return $this->redirectToRoute('proformaentreprises.add');
    }

    #[Route('/check-tva', name: "proformaentreprises.tvacheck", methods: ['GET'])]
    public function checkTvaAdd(Request $request): Response
    {
        $this->checkTva($request);
        return $this->redirectToRoute('proformaentreprises.add');
    }

    private function checkTva(Request $request): void
    {
        $session = $this->requestStack->getSession();
        if (!$session->has('tva')) $session->set('tva', 0);
        $active = $request->query->get('active');
        if ($active === "on") {
            $tva = 18;
            $session->set('tva', $tva);
            $this->flash->success("Tva activée.");
        } else {
            $session->remove('tva');
            $this->flash->success("Tva désactivée.");
        }
    }

    #[Route('/check-tva/edit/{id}', name: 'proformaentreprises.tvacheckedit', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function checkTvaEdit(Request $request, Proformaentreprises $proformaentreprises): Response
    {
        $this->checkTva($request);
        return $this->redirectToRoute('proformaentreprises.edit', ['id' => $proformaentreprises->getId()]);
    }

    #[Route('/check-remise', name: 'proformaentreprises.remisecheck', methods: ['GET'])]
    public function checkTRemiseAdd(Request $request): Response
    {
        $this->checkRemise($request);
        return $this->redirectToRoute('proformaentreprises.add');
    }

    private function checkRemise(Request $request): void
    {
        $session = $this->requestStack->getSession();
        if (!$session->has('remise')) $session->set('remise', 0);
        $remise = $request->query->getInt('remise');
        if ($remise && $remise > 0) {
            $valremise = $remise;
            $session->set('remise', $valremise);
            $this->flash->success("Remise activée.");
        } elseif ($remise === 0) {
            $session->remove('remise');
        } else {
            $session->remove('remise');
            $this->flash->success("Remise annulée.");
        }
    }

    #[Route('/check-remise/edit/{id}', name: 'proformaentreprises.remisecheckedit', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function checkRemiseEdit(Request $request, Proformaentreprises $proformaentreprises): Response
    {
        $this->checkRemise($request);
        return $this->redirectToRoute('proformaentreprises.edit', ['id' => $proformaentreprises->getId()]);
    }

    #[Route('/valider/proforma/{id}', name: 'valider.proforma', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function valider(Proformaentreprises $proformaentreprises): Response
    {
        $proformaentreprises->setValider(true);
        $proformaentreprises->setStatutproforma(2);
        $this->manager->setEntity($proformaentreprises, 'edit');
        $this->flash->success("Proforma validée.");
        return $this->redirectToRoute('proformaentreprises.index');
    }

    #[Route('/invalider/{id}', name: 'invalider.proforma', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function invalider(Proformaentreprises $proformaentreprises): Response
    {
        $proformaentreprises->setStatutproforma(0);
        $proformaentreprises->setValider(false);
        $this->manager->setEntity($proformaentreprises, 'edit');
        $this->flash->success("Proforma annulée.");
        return $this->redirectToRoute('proformaentreprises.index');
    }

    /**
     * @throws \Doctrine\DBAL\Exception
     */
    #[Route('/annuler', name: "proformaentreprises.annuler", methods: ['GET'])]
    public function annulerProforma(): Response
    {
        //On vide la table panier
        $connection = $this->entityManager->getConnection();
        $platform = $connection->getDatabasePlatform();
        $connection->executeUpdate($platform->getTruncateTableSQL('paniers_entreprise', true));
        $session = $this->requestStack->getSession();
        $session->remove('client');
        $session->remove('tva');
        $session->remove('remise');
        $this->flash->success("Prestation annulée.");
        return $this->redirectToRoute('proformaentreprises.add');
    }

    #[Route('/add', name: "proformaentreprises.add", methods: ['GET', 'POST'])]
    public function addProforma(HelpersInterfaces $helper, Request $request, PaniersEntrepriseRepository $paniersRepository): Response
    {
        //On reccupère la tva en session
        $session = $this->requestStack->getSession();
        if (!$session->has('tva')) $session->set('tva', 0);
        $tva = $session->get('tva');

        //On reccupère la remise en session
        if (!$session->has('remise')) $session->set('remise', 0);
        $remise = $session->get('remise');

        $form_proforma = $this->createForm(ProformaTitreDateType::class);
        $form_proforma->handleRequest($request);

        $panier = new PaniersEntreprise();
        $form = $this->createForm(PaniersEntrepriseType::class, $panier);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $pu = intval($form->get('pu')->getData() * $form->get('quantite')->getData());
            $panier->setPt($pu);
            $panier->setUsers($this->getUser());
            $this->manager->setEntity($panier, 'new');
            $this->flash->success("Produit ajouté avec succès.");
            return $this->redirectToRoute('proformaentreprises.add');
        }

        $tt = $paniersRepository->findSum();
        $tt = $tt['somme'];
        $laTva = (float)$tva;
        $laRemise = (float)$remise;
        $montantTva = ($tt * $laTva / 100);
        $montantRemise = ($tt + $montantTva) * ($laRemise / 100);
        $TTCFINAL = $tt + $montantTva - $montantRemise;
        $montantlettre = strtoupper($helper->towords($TTCFINAL));
        return $this->render('entreprises/proformas/add_proforma.html.twig', [
            'paniers' => $paniersRepository->findAll(),
            'totalHT' => $tt,
            'TTCFINAL' => $TTCFINAL,
            'form_proforma' => $form_proforma->createView(),
            'form' => $form->createView(),
            'panier' => $panier,
            'tva' => $tva,
            'remise' => $remise,
            'montantRemise' => $montantRemise,
            'montantTva' => $montantTva,
            'title' => 'Ajouter une nouvelle proforma',
            'libelle_liste' => 'Liste des proformas',
            'libelle_ajouter' => 'Ajouter des produits',
            'montantlettre' => $montantlettre
        ]);
    }

    #[Route('/save', name: "proformaentreprises.save", methods: ['GET', 'POST'])]
    public function saveProforma(Request $request): Response
    {
        $facture = new Proformaentreprises();
        $form = $this->createForm(ProformaTitreDateType::class, $facture);
        $form->handleRequest($request);

        //if ($form->isSubmitted() && $form->isValid()) {
        if ($form->isSubmitted()) {
            //On reccupère la tva en session
            $session = $this->requestStack->getSession();
            if (!$session->has('tva')) $session->set('tva', []);
            $tva = $session->get('tva');
            //On reccupère la remise en session
            if (!$session->has('remise')) $session->set('remise', []);
            $remise = $session->get('remise');
            //On crée un tableau de session du client vide
            //On crée la session de la facture
            if (!$session->has('proforma')) $session->set('proforma', []);
            $proforma = $session->get('proforma');

            //Definition de l'année (2 derniers chiffres)
            $annee = date('y');

            $countProformaForNewYear = $this->proformaentreprisesRepository->nombreProformaDeLAnnee();

            $nombre = 0;
            if ($countProformaForNewYear === 1) {
                $nombre = $nombre + $countProformaForNewYear;
            } else {
                $lastProforma = $this->proformaentreprisesRepository->getLastProforma(date('Y'));
                $nombre = $lastProforma?->getNumero();
                $nombre++;
            }

            $numerolabel = 'US ' . $nombre /*Faire un service*/ . '/' . $annee;

            $totalHT = 0;
            $totalTTC = 0;
            $paniers = $this->paniersEntrepriseRepository->findAll();
            foreach ($paniers as $panier) {
                $totalHT += $panier->getPt();

                $proforma['facture'][$panier->getId()] = [
                    'id' => $panier->getId(),
                    'quantite' => $panier->getQuantite(),
                    'designation' => $panier->getDesignation(),
                    'pu' => round($panier->getPu()),
                    'pt' => round($panier->getPu() * $panier->getQuantite())
                ];
            }
            $tva = (int)$tva;

            $montantTva = ($totalHT * $tva) / 100;
            $montantRemise = ($totalHT + $montantTva) * ($remise / 100);
            $totalTTC = $totalTTC + ($totalHT + $montantTva) - $montantRemise;
            $clientsentreprises = $form->get('clientsentreprises')->getData();

            $clientsentreprises = $this->clientsentreprisesRepository->findOneById($clientsentreprises->getId());

            if ($clientsentreprises == null) {
                throw $this->createNotFoundException('Aucun client trouvé');
            } else {
                $facture->setClientsentreprises($clientsentreprises);
            }

            //Preparation des données pour enregistrement
            $facture->setNumero($nombre);
            $facture->setNumerolabelle($numerolabel);
            $facture->setTotalht($totalHT);
            $facture->setClientsentreprises($clientsentreprises);

            $facture->setTotalttc($totalTTC);
            $facture->setResteAPayer($totalTTC);
            $facture->setTva($tva);
            $facture->setRemise($remise);
            $facture->setProduits($proforma);
            $facture->setValider(false);
            $facture->setUsers($this->getUser());
            $facture->setEtat(0);
            $facture->setStatut(false);
            $this->manager->setEntity($facture, 'new');

            $this->proformaentreprisesRepository->mettreClientAJour($facture->getId(), $clientsentreprises->getId());
            //On enverra une notification à l'administrateur (reccupérant l'id de la proforma nouvellement ajouter)

            //On vide la table panier
            $this->cleanPanierByUser();
            $this->flash->success("Facture Enregistrée avec sucès.");
            return $this->redirectToRoute('proformaentreprises.add');
        }

        return $this->redirectToRoute('proformaentreprises.add');
    }

    /**
     * @throws \Doctrine\DBAL\Exception
     */
    #[Route('/{id}', name: 'proformaentreprises.delete', requirements: ['id' => Requirement::DIGITS], methods: ['DELETE'])]
    public function delete(Request $request, Proformaentreprises $proformaentreprises): Response
    {
        if ($this->isCsrfTokenValid('delete' . $proformaentreprises->getId(), $request->request->get('_token'))) {
            $this->manager->setEntity($proformaentreprises, 'delete');
            $this->cleanSession();
            $this->flash->success("Proforma supprimée avec succès.");
        }
        return $this->redirectToRoute('proformaentreprises.index');
    }

    /**
     * @throws \Doctrine\DBAL\Exception
     */
    private function cleanSession(): void
    {
        $connection = $this->entityManager->getConnection();
        $platform = $connection->getDatabasePlatform();
        $connection->executeUpdate($platform->getTruncateTableSQL('paniers', true));
        $session = $this->requestStack->getSession();
        $session->remove('client');
        $session->remove('tva');
        $session->remove('remise');
    }

    #[Route('/delete/item/{id}', name: 'entreprises.panier.delete.item.add', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function deleteItemAdd(PaniersEntreprise $paniers): Response
    {
        $this->suppItemsPanier($paniers);
        $this->flash->success("Désignation supprimée du tableau avec succès.");
        return $this->redirectToRoute('proformaentreprises.add');
    }

    private function suppItemsPanier($paniers): void
    {
        $this->manager->setEntity($paniers, 'delete');
    }

    #[Route('/delete/item/{id}/{idproforma}',
        name: 'entreprises.panier.delete.item.edit',
        requirements: ['id' => Requirement::DIGITS, 'idproforma' => Requirement::DIGITS],
        methods: ['GET']
    )]
    public function deleteItemEdit(PaniersEntreprise $paniers, ProformaentreprisesRepository $proformaentreprisesRepository, ?int $idproforma = null): Response
    {
        $proforma = $proformaentreprisesRepository->findOneBy(['id' => $idproforma]);
        $this->suppItemsPanier($paniers);
        $this->flash->success("Désignation supprimée du tableau avec succès.");
        return $this->redirectToRoute('proformaentreprises.edit', ['id' => $proforma->getId()]);
    }

}
