<?php

namespace App\Repository;

use App\Entity\ClientsSearch;
use App\Entity\Proformaparticuliers;
use App\Entity\Searchs\SearchParticuliersPrestations;
use DateTime;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\Persistence\ManagerRegistry;
use Knp\Component\Pager\Pagination\PaginationInterface;
use Knp\Component\Pager\PaginatorInterface;

/**
 * @extends ServiceEntityRepository<Proformaparticuliers>
 */
class ProformaparticuliersRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry, private readonly PaginatorInterface $paginator)
    {
        parent::__construct($registry, Proformaparticuliers::class);
    }

    /**
     * @return Proformaparticuliers[]|null
     */
    public function findLastNumero(): ?array
    {
        return $this->createQueryBuilder('p')
            ->select('p.numero')
            ->orderBy('p.id', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getResult();
    }

    public function nombreProformaDuMois(): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->andWhere('p.mois = :mois')
            ->setParameter('mois', date('Y-m'))
            ->getQuery()
            ->getOneOrNullResult();
        return $result !== null ? (int)$result[1] : 0;
    }


    public function getLastProforma(): ?Proformaparticuliers
    {
        return $this->createQueryBuilder('p')
            ->where('p.mois = :mois')
            ->setParameter('mois', date('Y-m'))
            ->orderBy('p.numero', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param ClientsSearch $search
     * @return PaginationInterface
     */
    public function recherche(ClientsSearch $search): PaginationInterface
    {
        $query = $this->createQueryBuilder('p');
        if ($search->q !== null) {
            $query = $query->where('p.nom LIKE :critere')
                ->orWhere('p.telephone LIKE :critere')
                ->orWhere('p.adresse LIKE :critere')
                ->orWhere('p.email LIKE :critere')
                ->orWhere('p.numerolabelle LIKE :critere')
                ->setParameter('critere', '%' . $search->q . '%');
        }
        $query->orderBy('p.datecreation', 'DESC');
        return $this->paginator->paginate($query, $search->page, $search->limit);
    }

    public function findProformaByPeriode(SearchParticuliersPrestations $search): PaginationInterface
    {
        $query = $this->createQueryBuilder('p');
        if ($search->startAt !== null) {
            $query = $query->andWhere('p.datecreation >= :startAt')
                ->setParameter('startAt', $search->startAt->format('Y-m-d'));
        }

        if ($search->endAt !== null) {
            $query = $query->andWhere('p.datecreation <= :endAt')
                ->setParameter('endAt', $search->endAt->format('Y-m-d'));
        }

        if ($search->user !== null) {
            $query = $query->andWhere('p.users = :user')
                ->setParameter('user', $search->user);
        }

        $query
            ->orderBy('p.datecreation', 'DESC')
            ->getQuery()
            ->getResult();

        return $this->paginator->paginate($query, $search->page, $search->limit);
    }

    public function getProformaByPeriode($datedebut, $datefin): ?array
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.datecreation >= :datedebut')
            ->setParameter('datedebut', $datedebut)
            ->andWhere('p.datecreation <= :datefin')
            ->setParameter('datefin', $datefin)
            ->orderBy('p.datecreation', 'DESC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @return Proformaparticuliers[]|null
     */
    public function findProformas(): ?array
    {
        return $this->createQueryBuilder('p')
            ->orderBy('p.datecreation', 'DESC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @return Proformaparticuliers[]|null
     */
    public function findYesterday(): ?array
    {
        $date = new DateTime('now - 1day');
        return $this->createQueryBuilder('p')
            ->andWhere('p.createdAt < :date')
            ->setParameter('date', $date)
            ->orderBy('p.id', 'DESC')
            ->setMaxResults(3)
            ->getQuery()
            ->getResult();
    }

    /**
     * @return Proformaparticuliers[]|null
     */
    public function findFacturesJournaliers(): ?array
    {
        return $this->createQueryBuilder('p')
            ->select("p.datefacture as date, SUM(p.montantpaye) as payer, SUM(p.totalht) as total_ht, SUM(p.totalttc) as total_ttc")
            ->orderBy('date', 'DESC')
            ->setMaxResults(5)
            ->groupBy('p.datefacture')
            ->getQuery()
            ->getResult();
    }


    /**
     * @return Proformaparticuliers[]|null
     */
    public function findFacturesHebdomadaire(): ?array
    {
        return $this->createQueryBuilder('p')
            ->select('YEAR(p.datefacture) as year, WEEK(p.datefacture) as week, SUM(p.montantpaye) as payer, SUM(p.totalht + p.totalttc) as total')
            ->orderBy('year', 'DESC')
            ->addOrderBy('week', 'DESC')
            ->setMaxResults(5)
            ->groupBy('year, week')
            ->getQuery()
            ->getResult();
    }


    /**
     * @return Proformaparticuliers[]|null
     */
    public function findFacturesMensuel(): ?array
    {
        return $this->createQueryBuilder('p')
            ->select('YEAR(p.datefacture) as year, MONTH(p.datefacture) as month, SUM(p.montantpaye) as payer, SUM(p.totalht + p.totalttc) as total')
            ->orderBy('year', 'DESC')
            ->addOrderBy('month', 'DESC')
            ->setMaxResults(5)
            ->groupBy('year, month')
            ->getQuery()
            ->getResult();
    }


    /**
     * @return Proformaparticuliers[]|null
     */
    public function findToday(): ?array
    {
        $date = new DateTime('now - 1day');
        return $this->createQueryBuilder('p')
            ->andWhere('p.createdAt >= :date')
            ->setParameter('date', $date)
            ->orderBy('p.id', 'DESC')
            ->setMaxResults(3)
            ->getQuery()
            ->getResult();
    }

    public function findSumTotal()
    {
        return $this->createQueryBuilder('p')
            ->select('SUM(p.totalht + p.totalttc) as total')
            ->getQuery()
            ->getSingleResult();
    }

    /**
     * @return Proformaparticuliers[]|null
     */
    public function findLastCode(): ?array
    {
        return $this->createQueryBuilder('p')
            ->select('p.code')
            ->orderBy('p.id', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getResult();
    }

    /**
     * @param $search
     * @return Proformaparticuliers[]|null
     */
    public function findClient($search): ?array
    {
        return $this->createQueryBuilder('p')
            ->where('p.nom = :nom')
            ->setParameter('nom', $search)
            ->orWhere('p.codelabel = :code')
            ->setParameter('code', $search)
            ->orWhere('p.telephone = :telephone')
            ->setParameter('telephone', $search)
            ->getQuery()
            ->getResult();
    }

    /**
     * @param bool $status
     * @return Proformaparticuliers[]|null
     */
    public function getProformasStatus(bool $status): ?array
    {
        return $this->createQueryBuilder('p')
            ->where('p.statut = :status')
            ->setParameter('status', $status)
            ->orderBy('p.id', 'DESC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function countProformaParticuliersStatus(bool $status): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->where('p.valider = :status')
            ->setParameter('status', $status)
            ->getQuery()
            ->getOneOrNullResult();

        return $result !== null ? (int)$result[1] : 0;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function countProformaByStatus(bool $status): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->where('p.statut = :status')
            ->setParameter('status', $status)
            ->getQuery()
            ->getOneOrNullResult();

        return $result !== null ? (int)$result[1] : 0;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function countAll(): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->getQuery()
            ->getOneOrNullResult();

        return $result !== null ? (int)$result[1] : 0;
    }
}
