<?php declare(strict_types=1);

namespace App\Services\Helpers;

use App\Repository\ProformaentreprisesRepository;
use App\Repository\ProformaparticuliersRepository;
use NumberToWords\Exception\InvalidArgumentException;
use NumberToWords\Exception\NumberToWordsException;
use NumberToWords\NumberToWords;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

readonly class HelpersServices implements HelpersInterfaces
{
    public function __construct(
        private RequestStack                   $requestStack,
        private ProformaentreprisesRepository  $proformaentreprisesRepository,
        private ProformaparticuliersRepository $proformaparticuliersRepository,
    )
    {
    }

    /**
     * @param float $numberToConvert
     * @param string|null $devise
     * @return string
     * @throws InvalidArgumentException
     * @throws NumberToWordsException
     */
    public function tocurrency(float $numberToConvert, ?string $devise = 'XOF'): string
    {
        if (!$devise) {
            return '';
        }
        $montant = 0;
        if ($numberToConvert) {
            $montant = $montant + ($numberToConvert * 100);
        } else {
            $montant = $montant + $numberToConvert;
        }
        /** @var Request $request */
        $request = $this->requestStack->getCurrentRequest();
        $locale = $request->getLocale();
        $numberToWords = new NumberToWords();
        $currencyTransformer = $numberToWords->getCurrencyTransformer($locale);
        return $currencyTransformer->toWords(intval($montant), $devise);
    }

    /**
     * @param int $numberToConvert
     * @return string
     * @throws InvalidArgumentException
     * @throws NumberToWordsException
     */
    public function towords(int $numberToConvert): string
    {
        /** @var Request $request */
        $request = $this->requestStack->getCurrentRequest();
        $locale = $request->getLocale();
        $numberToWords = new NumberToWords();
        $numberTransformer = $numberToWords->getNumberTransformer($locale);
        return $numberTransformer->toWords($numberToConvert);
    }

    /**
     * @return array
     */
    public function generateNumeroProforma(): array
    {
        //Definition de l'année (2 derniers chiffres)
        $annee = date('y');

        $countProformaForNewYear = $this->proformaentreprisesRepository->nombreProformaDeLAnnee();

        $nombre = 0;
        if ($countProformaForNewYear === 1) {
            $nombre = $nombre + $countProformaForNewYear;
        } else {
            $lastProforma = $this->proformaentreprisesRepository->getLastProforma((int)date('Y'));
            $nombre = $lastProforma?->getNumero();
            $nombre++;
        }
        $numeroLabel = 'US ' . $nombre . '/' . $annee;
        return [$nombre, $numeroLabel];
    }

    /**
     * @return array
     */
    public function generateNumeroProformaParticulier(): array
    {
        //Definition de l'année (2 derniers chiffres)
        $mois = date('m');
        $countProformaForNewYear = $this->proformaparticuliersRepository->nombreProformaDuMois();

        $nombre = 0;
        if ($countProformaForNewYear === 1) {
            $nombre = $nombre + $countProformaForNewYear;
        } else {
            $lastProforma = $this->proformaparticuliersRepository->getLastProforma();
            $nombre = $lastProforma?->getNumero();
            $nombre++;
        }

        $numeroLabel = 'US ' . $nombre . '/' . $mois;
        return [$nombre, $numeroLabel];
    }
}