<?php

namespace App\Controller;

use App\Entity\Notes;
use App\Form\NotesType;
use App\Managers\Managers;
use App\Repository\NotesRepository;
use App\Services\Helpers\FlashServiceInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(new Expression('is_granted("ROLE_SUPER_ADMIN")'))]
class NotesController extends AbstractController
{
    public function __construct(
        private readonly Managers              $manager,
        private readonly NotesRepository       $notesRepository,
        private readonly FlashServiceInterface $flash,
    )
    {
    }

    #[Route('/notes/liste', name: "notes", options: ['expose' => true], methods: ['GET'])]
    public function index(): Response
    {
        $notes = $this->notesRepository->findAll();
        return $this->render('dashboard/notes/index.html.twig', [
            'notes' => $notes,
            'active' => 'notes',
            'title' => 'Liste des notes reçues',
        ]);
    }

    public function notesForm(Request $request): Response
    {
        //Formulaire pour ajouter une note
        $note = new Notes();
        $form = $this->createForm(NotesType::class, $note);
        $form->handleRequest($request);
        return $this->render('dashboard/includes/modal_quickNote.html.twig', [
            'note' => $note,
            'form_note' => $form->createView(),
        ]);
    }

    #[Route('/note/new', name: "note.new", options: ['expose' => true], methods: ['GET'])]
    public function new(Request $request): Response
    {
        if ($request->isXmlHttpRequest()) {
            $note = new Notes();
            $form = $this->createForm(NotesType::class, $note);
            $form->handleRequest($request);
            if ($form->isSubmitted() && $form->isValid()) {
                $note->setUsers($this->getUser());
                $this->manager->setEntity($note, 'new');
                $this->flash->success('La note à bien été envoyée');
                return new Response('success', 200);
            } else {
                return new Response('error', 200);
            }
        }
        return new Response('success', 200);
    }

    #[Route('/note/delete/{id}', name: "note.delete", requirements: ['id' => Requirement::DIGITS], options: ['expose' => true], methods: ['DELETE'])]
    public function delete(Request $request, Notes $note): Response
    {
        if ($this->isCsrfTokenValid('delete' . $note->getId(), $request->request->get('_token'))) {
            $this->manager->setEntity($note, 'new');
            $this->flash->success('La note à bien été supprimée');
        }

        return $this->redirectToRoute('notes');
    }
}
