<?php

namespace App\Controller;

use App\Entity\PaiementParticulier;
use App\Entity\PaniersParticulier;
use App\Entity\Proformaparticuliers;
use App\Entity\Searchs\SearchParticuliersPrestations;
use App\Form\Particuliers\PaniersParticulierType;
use App\Form\Particuliers\ProformaTitreDateType;
use App\Form\Particuliers\ProformaType;
use App\Form\Particuliers\Searchs\SearchParticuliersPrestationsType;
use App\Managers\ManagersInterface;
use App\Repository\PaniersParticulierRepository;
use App\Repository\ProformaparticuliersRepository;
use App\Services\Helpers\FlashServiceInterface;
use App\Services\Helpers\HelpersInterfaces;
use DateTime;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/admin/particuliers/prestations')]
#[IsGranted(new Expression('is_granted("ROLE_SUPER_ADMIN") or is_granted("ROLE_PARTICULIER")'))]
class PrestationsparticuliersController extends AbstractController
{
    public function __construct(
        private readonly PaniersParticulierRepository $paniersParticulierRepository,
        private readonly FlashServiceInterface        $flash,
        private readonly EntityManagerInterface       $entityManager,
        private readonly ManagersInterface            $manager,
        private readonly RequestStack                 $requestStack,
    )
    {
    }

    /**
     * @throws Exception
     */
    #[Route('/', name: "proformaparticuliers.index", methods: ['GET', 'POST'])]
    public function index(Request $request, ProformaparticuliersRepository $proformaparticuliersRepository): Response
    {
        $this->deletenewfactureDetailsSession();

        $search = new SearchParticuliersPrestations();
        $form = $this->createForm(SearchParticuliersPrestationsType::class, $search);
        $form->handleRequest($request);
        $search->page = $request->query->getInt('page', 1);
        $proformas = $proformaparticuliersRepository->findProformaByPeriode($search);
        if ($form->isSubmitted() && $form->isValid()) {
            $proformas = $proformaparticuliersRepository->findProformaByPeriode($search);
        }

        return $this->render('dashboard/particuliers/factures/index.html.twig', [
            'search' => $search,
            'proformaparticuliers' => $proformas,
            'title' => 'Liste des prestations',
            'libelle_ajouter' => 'Nouvelle prestation',
            'active' => 'proformaparticuliers.index',
            'form' => $form->createView(),
        ]);
    }

    private function deletenewfactureDetailsSession(): void
    {
        $session = $this->requestStack->getSession();
        $session->remove('newFactureParticulier');
    }

    /**
     * @throws Exception
     */
    #[Route('/nouveau-paiement/{id}', name: "proformaparticuliers.nouveau.paiement", requirements: ['id' => Requirement::DIGITS], methods: ['GET', 'POST'])]
    public function nouveauPaiement(Proformaparticuliers $proformaparticulier): Response
    {
        $montantPaiement = 0;
        $detailPaiement = '';
        if (isset($_POST['envoyer'])) {

            $montantPaiement = (int)$_POST['montantPaiement'];
            $detailPaiement = $_POST['detailPaiement'];
            $datePaiement = $_POST['datePaiement'];
            $datePaiement = preg_replace("#(/)#", '-', $datePaiement);
            $datePaiement = new DateTimeImmutable($datePaiement);

            if ($montantPaiement < 0 || $proformaparticulier->getResteAPayer() < $montantPaiement) {
                $this->flash->danger('Montant non valide.');
                return $this->redirectToRoute('proformaparticuliers.nouveau.paiement', array('id' => $proformaparticulier->getId()));

            }
            $paimentparticulier = new PaiementParticulier;
            $paimentparticulier->setProformaParticulier($proformaparticulier);
            $paimentparticulier->setUser($this->getUser());
            $paimentparticulier->setMontant($montantPaiement);
            $paimentparticulier->setDetail($detailPaiement);
            $paimentparticulier->setLaDate($datePaiement);
            $proformaparticulier->setMontantPaye($proformaparticulier->getMontantPaye() + $montantPaiement);
            $proformaparticulier->setResteAPayer($proformaparticulier->getResteAPayer() - $montantPaiement);
            if ($proformaparticulier->getResteAPayer() == 0) {
                $proformaparticulier->setRecupere(true);
            }
            $this->flash->success("Paiement effectué.");
            $this->manager->setEntity($paimentparticulier, 'new');
        }

        return $this->render('dashboard/particuliers/factures/nouveau_paiment.html.twig', [
            'proformaparticulier' => $proformaparticulier,
            'title' => 'Nouveau paiement',
            'libelle_ajouter' => 'Nouvelle prestation',
            'active' => 'proformaparticuliers.index',
        ]);
    }

    #[Route('/new_details', name: "factureparticuliers.detail.new", methods: ['GET'])]
    public function detailNewFacture(): Response
    {
        $session = $this->requestStack->getSession();
        if (!$session->has('newFactureParticulier')) $session->set('newFactureParticulier', []);
        $facture = $session->get('newFactureParticulier');
        if (!$facture)
            return $this->redirectToRoute('proformaparticuliers.index');

        return $this->render('dashboard/particuliers/factures/details_new_facture.html.twig', [
            'facture' => $facture,
            'title' => 'Détail de la nouvelle prestation ajoutée',
        ]);
    }

    #[Route('/show/{id}', name: 'proformaparticuliers.show', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function show(Proformaparticuliers $proformaparticuliers): Response
    {
        $this->deletenewfactureDetailsSession();
        $roleUser = $this->isGranted('ROLE_SUPER_ADMIN');
        if ($roleUser && $proformaparticuliers->getStatut() === false) {
            $proformaparticuliers->setStatut(true);
            $this->manager->setEntity($proformaparticuliers, 'edit');
        }
        return $this->render('dashboard/particuliers/factures/show.html.twig', [
            'proformaparticulier' => $proformaparticuliers,
            'title' => 'Prestation',
            'active' => 'proformaparticuliers.show',
        ]);
    }

    /**
     * @throws Exception
     */
    #[Route('/{id}/edit', name: 'proformaparticuliers.edit', requirements: ['id' => Requirement::DIGITS], methods: ['GET', 'POST'])]
    public function edit(Request $request, Proformaparticuliers $proformaparticuliers, PaniersParticulierRepository $paniersRepository): Response
    {
        $this->deletenewfactureDetailsSession();
        $session = $this->requestStack->getSession();
        //On reccupère la tva en session
        if (!$session->has('tva')) $session->set('tva', []);
        $tva = $session->get('tva');
        $products = $proformaparticuliers->getProduits()['facture'];
        $dateDeLaFacture = $proformaparticuliers->getDateFacture();
        if ($paniersRepository->findAll() == null) {
            foreach ($products as $product) {
                $panier = new PaniersParticulier();
                $panier->setDesignation($product['designation']);
                $panier->setQuantite($product['quantite']);
                $panier->setPu($product['pu']);
                $panier->setPt($product['pt']);
                $panier->setUsers($this->getUser());
                $panier->setProformaid($proformaparticuliers->getId());
                $this->manager->setEntity($panier, 'new');
            }
        }

        $form_proforma = $this->createForm(ProformaType::class, $proformaparticuliers);
        $form_proforma->handleRequest($request);
        if ($form_proforma->isSubmitted() && $form_proforma->isValid()) {
            $dateconvert = date("Y-m-d H:i:s", strtotime($form_proforma->get('datefacture')->getData()));
            $date = new DateTime($dateconvert);
            $proformaparticuliers->setUsers($this->getUser());
            $proformaparticuliers->setDatecreation($date);
            $this->manager->setEntity($proformaparticuliers, 'edit');
            $this->cleanPanierByUser();
            $this->flash->success("Proforma modifiée avec succès.");
            return $this->redirectToRoute('proformaparticuliers.edit', ['id' => $proformaparticuliers->getId()]);
        }

        $panier = new PaniersParticulier();
        $form = $this->createForm(PaniersParticulierType::class, $panier);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $pu = intval($form->get('pu')->getData() * $form->get('quantite')->getData());
            $panier->setPt($pu);
            $panier->setUsers($this->getUser());
            $panier->setProformaid($proformaparticuliers->getId());
            $this->manager->setEntity($panier, 'new');
            $this->flash->success("Produit ajouté avec succès.");
            return $this->redirectToRoute('proformaparticuliers.edit', ['id' => $proformaparticuliers->getId()]);
        }

        return $this->render('dashboard/particuliers/factures/edit.html.twig', [
            'proformaparticulier' => $proformaparticuliers,
            'paniers' => $paniersRepository->findAll(),
            'totalHT' => $paniersRepository->findSum(),
            'form_proforma' => $form_proforma->createView(),
            'form' => $form->createView(),
            'tva' => $tva,
            'panier' => $panier,
            'title' => 'Editer client / prestation',
            'libelle_ajouter' => 'Nouvelle prestation',
            'libelle_liste' => 'Liste des prestations',
            'dateDeLaFacture' => $dateDeLaFacture
        ]);
    }

    private function cleanPanierByUser(): void
    {
        $paniers = $this->paniersParticulierRepository->findBy(['users' => $this->getUser()]);
        $session = $this->requestStack->getSession();
        foreach ($paniers as $panier) {
            $this->manager->setEntity($panier, 'delete');
        }
        $session->remove('client');
        $session->remove('tva');
    }

    #[Route('/products/edit/{id}', name: 'proformaparticuliers.editproducts', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function editProducts(Proformaparticuliers $proformaparticuliers, PaniersParticulierRepository $paniersRepository): Response
    {
        $this->deletenewfactureDetailsSession();
        $session = $this->requestStack->getSession();
        //On reccupère la tva en session
        if (!$session->has('tva')) $session->set('tva', []);
        $tva = $session->get('tva');

        //On crée la session de la facture
        if (!$session->has('proforma')) $session->set('proforma', []);
        $proforma = $session->get('proforma');

        $paniers = $paniersRepository->findAll();

        if (!$paniers) {
            $this->flash->danger("Aucun produit dans le panier.");
            return $this->redirectToRoute('proformaparticuliers.edit', ['id' => $proformaparticuliers->getId()]);
        }

        $totalHT = 0;
        $totalTTC = 0;

        foreach ($paniers as $panier) {
            $totalHT += $panier->getPt();

            $proforma['facture'][$panier->getId()] = [
                'id' => $panier->getId(),
                'quantite' => $panier->getQuantite(),
                'designation' => $panier->getDesignation(),
                'pu' => round($panier->getPu()),
                'pt' => round($panier->getPu() * $panier->getQuantite())
            ];
        }
        $tva = (float)$tva;
        $montantTva = ($totalHT * $tva) / 100;
        $totalTTC = ceil($totalTTC + ($totalHT + $montantTva));

        $proformaparticuliers->setTotalht($totalHT);
        $proformaparticuliers->setTotalttc($totalTTC);
        $proformaparticuliers->setTva($tva);
        $proformaparticuliers->setUsers($this->getUser());
        $proformaparticuliers->setProduits($proforma);
        $this->manager->setEntity($proformaparticuliers, 'edit');
        $this->flash->success("Produit(s) mis à jour avec succès.");
        $this->cleanPanierByUser();
        return $this->redirectToRoute('proformaparticuliers.index');
    }

    #[Route('/edit-client', name: "proformaparticuliers.otherClient", methods: ['GET'])]
    public function otherClient(): Response
    {
        $session = $this->requestStack->getSession();
        if (!$session->has('client')) $session->set('client', []);
        $session->remove('client');
        return $this->redirectToRoute('proformaparticuliers.add');
    }

    #[Route('/check-tva', name: "proformaparticuliers.tvacheck", methods: ['GET'])]
    public function checkTvaAdd(Request $request): Response
    {
        $this->checkTva($request);
        return $this->redirectToRoute('proformaparticuliers.add');
    }

    private function checkTva(Request $request): void
    {
        $session = $this->requestStack->getSession();
        if (!$session->has('tva')) $session->set('tva', 0);
        $active = $request->query->get('active');
        if ($active === "on") {
            $tva = 18;
            $session->set('tva', $tva);
            $this->flash->success("Tva activée.");
        } else {
            $session->remove('tva');
            $this->flash->success("Tva désactivée.");
        }
    }

    #[Route('/check-tva/edit/{id}', name: 'proformaparticuliers.tvacheckedit', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function checkTvaEdit(Request $request, Proformaparticuliers $proformaparticuliers): Response
    {
        $this->checkTva($request);
        return $this->redirectToRoute('proformaparticuliers.edit', ['id' => $proformaparticuliers->getId()]);
    }

    /**
     * @throws \Doctrine\DBAL\Exception
     */
    #[Route('/annuler', name: "proformaparticuliers.annuler", methods: ['GET'])]
    public function annulerProforma(): Response
    {
        //On vide la table panier
        $connection = $this->entityManager->getConnection();
        $platform = $connection->getDatabasePlatform();
        $connection->executeUpdate($platform->getTruncateTableSQL('paniers_particulier', true));
        $session = $this->requestStack->getSession();
        $session->remove('client');
        $session->remove('tva');
        $this->flash->success("Prestation annulée.");
        return $this->redirectToRoute('proformaparticuliers.add');
    }

    #[Route('/add', name: "proformaparticuliers.add", methods: ['GET', 'POST'])]
    public function addProforma(HelpersInterfaces $helper, Request $request, PaniersParticulierRepository $paniersRepository): Response
    {
        $montantlettre = '';
        $panier = new PaniersParticulier();
        $form = $this->createForm(PaniersParticulierType::class, $panier);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $pu = intval($form->get('pu')->getData() * $form->get('quantite')->getData());
            $panier->setPt($pu);
            $panier->setUsers($this->getUser());
            $this->manager->setEntity($panier, 'new');
            $this->flash->success("Produit ajouté avec succès.");
            return $this->redirectToRoute('proformaparticuliers.add');
        }

        $form_proforma = $this->createForm(ProformaTitreDateType::class);
        $form_proforma->handleRequest($request);

        $tt = $paniersRepository->findSum();
        $tt = $tt['somme'];
        if ($tt > 0) {
            $montantlettre = strtoupper($helper->towords($tt));
        }
        return $this->render('dashboard/particuliers/factures/add_facture.html.twig', [
            'paniers' => $paniersRepository->findAll(),
            'totalHT' => $tt,
            'form_proforma' => $form_proforma->createView(),
            'form' => $form->createView(),
            'panier' => $panier,
            'title' => 'Ajouter une nouvelle prestation',
            'libelle_liste' => 'Liste des prestations',
            'libelle_ajouter' => 'Ajouter des produits',
            'montantlettre' => $montantlettre,
            'montantTTC' => $tt
        ]);
    }

    #[Route('/save', name: "proformaparticuliers.save", methods: ['GET', 'POST'])]
    public function saveProforma(ProformaparticuliersRepository $proformaparticuliersRepository, Request $request): Response
    {
        $facture = new Proformaparticuliers();
        $form = $this->createForm(ProformaTitreDateType::class, $facture);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            //On reccupère la tva en session
            $session = $this->requestStack->getSession();
            if (!$session->has('tva')) $session->set('tva', []);
            $tva = $session->get('tva');
            //On crée la session de la facture
            if (!$session->has('proforma')) $session->set('proforma', []);
            $proforma = $session->get('proforma');

            //Versement
            $montantPaye = (int)$form->get('montantpaye')->getData();
            $resteAPayer = (int)$form->get('resteAPayer')->getData();

            if ($resteAPayer < 0) {
                $this->flash->danger("Montant payé supérieur au montant total.");
                return $this->redirectToRoute('proformaparticuliers.add');
            }

            $paiementparticulier = new Paiementparticulier;
            if ($montantPaye > 0) {
                $paiementparticulier->setProformaParticulier($facture);
                $paiementparticulier->setMontant($montantPaye);
                $paiementparticulier->setUser($this->getUser());
            }

            //Definition de l'année (2 derniers chiffres)
            $mois = date('m');
            $countProformaForNewYear = $proformaparticuliersRepository->nombreProformaDuMois();

            $nombre = 0;
            if ($countProformaForNewYear === 1) {
                $nombre = $nombre + $countProformaForNewYear;
            } else {
                $lastProforma = $proformaparticuliersRepository->getLastProforma();
                $nombre = $lastProforma?->getNumero();
                $nombre++;
            }

            $numerolabel = 'US ' . $nombre /*Faire un service*/ . '/' . $mois;

            $totalHT = 0;
            $totalTTC = 0;

            $paniers = $this->paniersParticulierRepository->findAll();
            foreach ($paniers as $panier) {
                $totalHT += $panier->getPt();

                $proforma['facture'][$panier->getId()] = [
                    'id' => $panier->getId(),
                    'quantite' => $panier->getQuantite(),
                    'designation' => $panier->getDesignation(),
                    'pu' => round($panier->getPu()),
                    'pt' => round($panier->getPu() * $panier->getQuantite())
                ];
            }
            $tva = (int)$tva;
            $montantTva = ($totalHT * $tva) / 100;
            $totalTTC = $totalTTC + ($totalHT + $montantTva);


            //Definition de l'année (2 derniers chiffres)
            $jour_mois = date('d/m');
            $codelabel = $nombre /*Faire un service*/ . '/' . $jour_mois;

            $facture->setUsers($this->getUser());
            $facture->setCode($nombre);
            $facture->setCodelabel($codelabel);

            //Preparation des données pour enregistrement
            $facture->setNumero($nombre);
            $facture->setNumerolabel($numerolabel);
            $facture->setTotalht($totalHT);
            $facture->setTotalttc($totalTTC);
            $facture->setTva($tva);
            $facture->setProduits($proforma);
            $facture->setValider(false);
            $facture->setUsers($this->getUser());
            $facture->setEtat(0);
            $facture->setStatut(1);
            $this->manager->setEntity($facture, 'new');
            if ($montantPaye > 0) {
                $this->manager->setEntity($paiementparticulier, 'new');
            }

            if (!$session->has('newFactureParticulier'))
                $session->set('newFactureParticulier', $facture);
            //On enverra une notification à l'administrateur (reccupérant l'id de la proforma nouvellement ajouter)

            //On vide la table panier
            $this->cleanPanierByUser();
            $this->flash->success("Prestation Enregistrée avec sucès.");
            return $this->redirectToRoute('factureparticuliers.detail.new');
        }
        return $this->redirectToRoute('factureparticuliers.detail.new');
    }

    #[Route('/{id}', name: 'proformaparticuliers.delete', requirements: ['id' => Requirement::DIGITS], methods: ['DELETE'])]
    public function delete(Request $request, Proformaparticuliers $proformaparticuliers): Response
    {
        if ($this->isCsrfTokenValid('delete' . $proformaparticuliers->getId(), $request->request->get('_token'))) {
            $this->manager->setEntity($proformaparticuliers, 'delete');
            $this->flash->success("Prestation supprimée avec succès.");
        }
        return $this->redirectToRoute('proformaparticuliers.index');
    }

    #[Route('/annuler/{id}', name: 'proformaparticuliers.annuler.prestation', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function annulerPrestation(Proformaparticuliers $proformaparticuliers): Response
    {
        $proformaparticuliers->setAnnuler(true);
        $this->manager->setEntity($proformaparticuliers, 'edit');
        $this->flash->success("Prestation annulée avec succès.");
        return $this->redirectToRoute('proformaparticuliers.index');
    }

    #[Route('/recuperer/{id}', name: 'proformaparticuliers.recuperer.prestation', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function recupererPrestation(Proformaparticuliers $proformaparticuliers): Response
    {
        $proformaparticuliers->setRecupere(true);
        $this->manager->setEntity($proformaparticuliers, 'edit');
        $this->flash->success("Pièce récupérée avec succès.");
        return $this->redirectToRoute('proformaparticuliers.index');
    }

    #[Route('/delete/item/{id}', name: 'particuliers.panier.delete.item.add', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function deleteItemAdd(PaniersParticulier $paniers): Response
    {
        $this->suppItemsPanier($paniers);
        $this->flash->success("Désignation supprimée du tableau avec succès.");
        return $this->redirectToRoute('proformaparticuliers.add');
    }

    private function suppItemsPanier($paniers): void
    {
        $this->manager->setEntity($paniers, 'delete');
    }

    #[Route('/delete/item/{id}/{idproforma}',
        name: 'particuliers.panier.delete.item.edit',
        requirements: ['id' => Requirement::DIGITS, 'idproforma' => Requirement::DIGITS],
        methods: ['GET']
    )]
    public function deleteItemEdit(PaniersParticulier $paniers, ProformaparticuliersRepository $proformaparticuliersRepository, ?int $idproforma = null): Response
    {
        $proforma = $proformaparticuliersRepository->findOneBy(['id' => $idproforma]);
        $this->suppItemsPanier($paniers);
        $this->flash->success("Désignation supprimée du tableau avec succès.");
        return $this->redirectToRoute('proformaparticuliers.edit', ['id' => $proforma->getId()]);
    }

    /**
     * @throws \Doctrine\DBAL\Exception
     */
    private function cleanSession(): void
    {
        $connection = $this->entityManager->getConnection();
        $platform = $connection->getDatabasePlatform();
        $connection->executeUpdate($platform->getTruncateTableSQL('paniers', true));
        $session = $this->requestStack->getSession();
        $session->remove('client');
        $session->remove('tva');
        $this->deletenewfactureDetailsSession();
    }

}
