<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\Form;

use Symfony\Component\Form\Exception\InvalidArgumentException;

/**
 * A form extension with preloaded types, type extensions and type guessers.
 *
 * @author Bernhard Schussek <bschussek@gmail.com>
 */
class PreloadedExtension implements FormExtensionInterface
{
    private array $types = [];

    /**
     * Creates a new preloaded extension.
     *
     * @param FormTypeInterface[]            $types          The types that the extension should support
     * @param FormTypeExtensionInterface[][] $typeExtensions The type extensions that the extension should support
     */
    public function __construct(
        array $types,
        private array $typeExtensions,
        private ?FormTypeGuesserInterface $typeGuesser = null,
    ) {
        foreach ($types as $type) {
            $this->types[$type::class] = $type;
        }
    }

    public function getType(string $name): FormTypeInterface
    {
        if (!isset($this->types[$name])) {
            throw new InvalidArgumentException(sprintf('The type "%s" cannot be loaded by this extension.', $name));
        }

        return $this->types[$name];
    }

    public function hasType(string $name): bool
    {
        return isset($this->types[$name]);
    }

    public function getTypeExtensions(string $name): array
    {
        return $this->typeExtensions[$name]
            ?? [];
    }

    public function hasTypeExtensions(string $name): bool
    {
        return !empty($this->typeExtensions[$name]);
    }

    public function getTypeGuesser(): ?FormTypeGuesserInterface
    {
        return $this->typeGuesser;
    }
}
