<?php

namespace App\Repository;

use App\Entity\ClientsSearch;
use App\Entity\Proformaentreprises;
use App\Entity\Searchs\SearchEntrepriesProforma;
use DateTime;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\Persistence\ManagerRegistry;
use Knp\Component\Pager\Pagination\PaginationInterface;
use Knp\Component\Pager\PaginatorInterface;

/**
 * @extends ServiceEntityRepository<Proformaentreprises>
 */
class ProformaentreprisesRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry, private readonly PaginatorInterface $paginator)
    {
        parent::__construct($registry, Proformaentreprises::class);
    }

    public function nombreProformaDeLAnnee(): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->where('p.annee = :annee')
            ->setParameter('annee', date('Y'))
            ->getQuery()
            ->getOneOrNullResult();
        return $result !== null ? (int)$result[1] : 0;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function getLastProforma(int $annee): ?Proformaentreprises
    {
        return $this->createQueryBuilder('p')
            ->where('p.annee = :annee')
            ->setParameter('annee', $annee)
            ->orderBy('p.numero', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @param ClientsSearch $search
     * @return PaginationInterface
     */
    public function recherche(ClientsSearch $search): PaginationInterface
    {
        $query = $this->createQueryBuilder('p')
            ->join('p.clientsentreprises', 'cp')
            ->addSelect('cp');
        if ($search->q !== null) {
            $query = $query
                ->where('cp.nom LIKE :critere')
                ->orWhere('p.numerolabelle LIKE :critere')
                ->setParameter('critere', '%' . $search->q . '%');
        }
        $query->orderBy('p.datecreation', 'DESC');
        return $this->paginator->paginate($query, $search->page, $search->limit);
    }

    public function mettreClientAJour($proforma, $entreprise): void
    {
        $qb = $this->createQueryBuilder('p');
        $qb->update(Proformaentreprises::class, 'p')
            ->set('p.clientsentreprises', '?1')
            ->where('p.id = ?2')
            ->setParameter(1, $entreprise)
            ->setParameter(2, $proforma);

        $qb = $qb->getQuery();
        $qb->execute();
    }

    /**
     * @return Proformaentreprises[]|null
     */
    public function findLastNumero(): ?array
    {
        return $this->createQueryBuilder('p')
            ->select('p.numero')
            ->orderBy('p.id', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getResult();
    }

    /**
     * @param $datedebut
     * @param $datefin
     * @return Proformaentreprises[]|null
     */
    public function findProformaByPeriode2($datedebut, $datefin): ?array
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.datecreation >= :datedebut')
            ->setParameter('datedebut', $datedebut)
            ->andWhere('p.datecreation <= :datefin')
            ->setParameter('datefin', $datefin)
            ->orderBy('p.id', 'DESC')
            ->getQuery()
            ->getResult();
    }

    public function findProformaByPeriode(SearchEntrepriesProforma $search): PaginationInterface
    {
        $query = $this->createQueryBuilder('p');
        if ($search->startAt !== null) {
            $query = $query->andWhere('p.datecreation >= :startAt')
                ->setParameter('startAt', $search->startAt);
        }

        if ($search->endAt !== null) {
            $query = $query->andWhere('p.datecreation <= :endAt')
                ->setParameter('endAt', $search->endAt);
        }

        if ($search->user !== null) {
            $query = $query->andWhere('p.users = :user')
                ->setParameter('user', $search->user);
        }

        $query->orderBy('p.id', 'DESC')->getQuery();

        return $this->paginator->paginate($query, $search->page, $search->limit);
    }

    /**
     * @param $datedebut
     * @param $datefin
     * @return Proformaentreprises[]|null
     */
    public function getProformaByPeriode($datedebut, $datefin): ?array
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.datecreation >= :datedebut')
            ->setParameter('datedebut', $datedebut)
            ->andWhere('p.datecreation <= :datefin')
            ->setParameter('datefin', $datefin)
            ->orderBy('p.id', 'DESC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @return Proformaentreprises[]|null
     */
    public function findFactures(): ?array
    {
        return $this->createQueryBuilder('p')
            ->where('p.attacheboncommande IS NOT NULL ')
            ->orderBy('p.id', 'DESC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @return Proformaentreprises[]|null
     */
    public function findProformas(): ?array
    {
        return $this->createQueryBuilder('p')
            ->where('p.attacheboncommande IS NULL ')
            ->orderBy('p.id', 'DESC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @return Proformaentreprises[]|null
     */
    public function findYesterday(): ?array
    {
        $date = new DateTime('now - 1day');
        return $this->createQueryBuilder('p')
            ->andWhere('p.createdAt < :date')
            ->setParameter('date', $date)
            ->orderBy('p.id', 'DESC')
            ->setMaxResults(3)
            ->getQuery()
            ->getResult();
    }

    /**
     * @return Proformaentreprises[]|null
     */
    public function findToday(): ?array
    {
        $date = new DateTime('now - 1day');
        return $this->createQueryBuilder('p')
            ->andWhere('p.createdAt >= :date')
            ->setParameter('date', $date)
            ->orderBy('p.id', 'DESC')
            ->setMaxResults(3)
            ->getQuery()
            ->getResult();
    }

    public function findSumTotal()
    {
        return $this->createQueryBuilder('p')
            ->select('SUM(p.totalht + p.totalttc) as total')
            ->getQuery()
            ->getSingleResult();
    }

    /**
     * @param bool $status
     * @return Proformaentreprises[]|null
     */
    public function getProformaStatus(bool $status): ?array
    {
        return $this->createQueryBuilder('p')
            ->where('p.statut = :status')
            ->setParameter('status', $status)
            ->orderBy('p.id', 'DESC')
            ->getQuery()
            ->getResult();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function countProformaEntreprisesStatus(bool $status): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->where('p.valider = :status')
            ->setParameter('status', $status)
            ->getQuery()
            ->getOneOrNullResult();

        return $result !== null ? (int)$result[1] : 0;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function countProformaByStatus(bool $status): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->where('p.statut = :status')
            ->setParameter('status', $status)
            ->getQuery()
            ->getOneOrNullResult();

        return $result !== null ? (int)$result[1] : 0;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function countAll(): int
    {
        $result = $this->createQueryBuilder('p')
            ->select('COUNT(p.id)')
            ->getQuery()
            ->getOneOrNullResult();

        return $result !== null ? (int)$result[1] : 0;
    }
}
