<?php

namespace App\Controller\Particuliers;

use App\Services\Carts\CartItem;
use App\Services\Carts\CartItemType;
use App\Services\Carts\CartServiceInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/dashboard/particulier/cart')]
#[IsGranted(new Expression('is_granted("ROLE_SUPER_ADMIN") or is_granted("ROLE_PARTICULIER")'))]
class CartController extends AbstractController
{
    public function __construct(private readonly CartServiceInterface $cart)
    {
    }

    #[Route('/', name: 'particulier.cart.index')]
    public function index(Request $request): Response
    {
        /** @var CartItem[] $carts */
        $carts = $this->cart->getItems();
        $cart = new CartItem('', 1, 0.0);
        $form = $this->createForm(CartItemType::class, $cart)->handleRequest($request);
        return $this->render('particuliers/cart/index.html.twig', [
            'cart' => $cart,
            'carts' => $carts,
            'cartService' => $this->cart,
            'form' => $form->createView(),
            'title' => 'Ajouter une proforma',
            'libelle_liste' => 'Listes des proformas',
        ]);
    }

    #[Route('/add', name: 'particulier.cart.add')]
    public function add(Request $request): Response
    {
        $cartItem = new CartItem('', 1, 0.0);
        $form = $this->createForm(CartItemType::class, $cartItem);
        $form->handleRequest($request);
        $referer = $request->headers->get('referer');
        if ($form->isSubmitted() && $form->isValid()) {
            $this->cart->add($cartItem);
            $this->addFlash('success', 'Article ajouté au panier !');
            return new RedirectResponse($referer);
        }
        return new RedirectResponse($referer);
    }

    #[Route('/update/{index}', name: 'particulier.cart.update')]
    public function updateItem(int $index, Request $request): Response
    {
        $designation = $request->request->get('designation');
        $quantity = (int)$request->request->get('quantity');
        $unitPrice = (float)$request->request->get('unitPrice');

        $this->cart->updateItem($index, $designation, $quantity, $unitPrice);
        $this->addFlash('success', 'Article mis à jour !');

        $referer = $request->headers->get('referer');
        return new RedirectResponse($referer);
    }

    #[Route('/update-multiple', name: 'particulier.cart.update.all')]
    public function updateMultipleItems(Request $request): Response
    {
        /** @var array $updatedItems */
        $updatedItems = $request->request->get('updatedItems', []);
        $this->cart->updateItems($updatedItems);
        $this->addFlash('success', 'Panier mis à jour avec succès !');

        $referer = $request->headers->get('referer');
        return new RedirectResponse($referer);
    }

    #[Route('/remove/{index}', name: 'particulier.cart.remove')]
    public function remove(int $index, Request $request): Response
    {
        $cartItems = $this->cart->getItems();
        if (isset($cartItems[$index])) {
            if (count($cartItems) > 1) {
                $this->cart->remove($index);
            } else {
                $this->cart->clear();
            }
            $this->addFlash('success', 'Article retiré du panier !');
        } else {
            $this->addFlash('error', 'L\'article n\'existe pas dans le panier.');
        }
        $this->addFlash('success', 'Article retiré du panier !');
        $referer = $request->headers->get('referer');
        return new RedirectResponse($referer);
    }

    #[Route('/clear', name: 'particulier.cart.clear')]
    public function clear(Request $request): Response
    {
        $this->cart->clear();
        $this->addFlash('success', 'La proforma a bien été annulée !');
        $referer = $request->headers->get('referer');
        return new RedirectResponse($referer);
    }
}