<?php declare(strict_types=1);

namespace App\Services\Carts;


use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

readonly class CartService implements CartServiceInterface
{
    private const CART_KEY = 'cart_items';
    private const CART_TVA = 'cart_tva';
    private const CART_DISCOUNT = 'cart_discount';

    public function __construct(private RequestStack $request)
    {
    }

    public function add(CartItem $item): void
    {
        $cartItems = $this->session()->get(self::CART_KEY, []);
        $cartItems[] = $item;
        $this->session()->set(self::CART_KEY, $cartItems);
    }

    private function session(): SessionInterface
    {
        return $this->request->getSession();
    }

    public function getItems(): array
    {
        return $this->session()->get(self::CART_KEY, []);
    }

    public function update(int $index, string $designation, int $quantity, float $price): void
    {
        $cartItems = $this->session()->get(self::CART_KEY, []);

        if (isset($cartItems[$index])) {
            $cartItems[$index] = new CartItem($designation, $quantity, $price);
            $this->session()->set(self::CART_KEY, $cartItems);
        }
    }

    public function updateItems(array $updatedItems): void
    {
        $cartItems = $this->session()->get(self::CART_KEY, []);

        foreach ($updatedItems as $index => $newData) {
            if (isset($cartItems[$index])) {
                $cartItems[$index] = new CartItem(
                    $newData['designation'] ?? $cartItems[$index]->getDesignation(),
                    $newData['quantity'] ?? $cartItems[$index]->getQuantity(),
                    $newData['price'] ?? $cartItems[$index]->getPrice()
                );
            }
        }

        $this->session()->set(self::CART_KEY, $cartItems);
    }

    /**
     * Calcule le montant de la TVA sur le total du panier.
     *
     * @param int $taxRate Taux de TVA en pourcentage (par exemple, 20 pour 20%)
     * @return float
     */
    public function calculateTax(int $taxRate): float
    {
        $total = $this->getTotalHorsTaxe(); // Total avant taxes
        // Met à jour la TVA dans la session si nécessaire
        $this->session()->set(self::CART_TVA, $taxRate);

        $montantHtAfterRemise = $total - $this->calculateDiscount($this->discount());

        return ($montantHtAfterRemise * $taxRate) / 100; // Retourne le montant de la taxe
    }

    /**
     * Calcule le total du panier HT.
     *
     * @return float
     */
    public function getTotalHorsTaxe(): float
    {
        $cartItems = $this->session()->get(self::CART_KEY, []);
        $total = 0.0;

        // Calcule le total de base (HT) en parcourant tous les articles du panier
        /** @var CartItem[] $cartItems */
        foreach ($cartItems as $item) {
            $total += $item->getQuantity() * $item->getPrice();
        }
        return $total;
    }

    /**
     * Calcule le montant de la remise sur le total du panier.
     *
     * @param int $discountRate Taux de remise en pourcentage (par exemple, 10 pour 10%)
     * @return float
     */
    public function calculateDiscount(int $discountRate): float
    {
        $total = $this->getTotalHorsTaxe();
        $this->session()->set(self::CART_DISCOUNT, $discountRate);
        return ($total * $discountRate) / 100;
    }

    /**
     * @return int
     */
    public function discount(): int
    {
        return $this->session()->get(self::CART_DISCOUNT, 0);
    }

    /**
     * Calcule le total du panier en prenant en compte la remise et la TVA.
     *
     * @return float
     */
    public function getTotal(): float
    {
        $cartItems = $this->session()->get(self::CART_KEY, []);
        $total = 0.0;

        // Calcule le total de base (HT) en parcourant tous les articles du panier
        /** @var CartItem[] $cartItems */
        foreach ($cartItems as $item) {
            $total += $item->getQuantity() * $item->getPrice();
        }

        // Applique la remise, si elle existe
        $discountRate = $this->discount(); // taux de remise en pourcentage
        if ($discountRate > 0) {
            $total -= ($total * $discountRate) / 100;
        }

        // Applique la TVA, si elle existe
        $taxRate = $this->tva(); // taux de TVA en pourcentage
        if ($taxRate > 0) {
            $total += ($total * $taxRate) / 100;
        }

        return $total;
    }

    /**
     * @return int
     */
    public function tva(): int
    {
        return $this->session()->get(self::CART_TVA, 0);
    }

    public function clear(): void
    {
        $this->session()->remove(self::CART_KEY);
        $this->session()->remove(self::CART_TVA);
        $this->session()->remove(self::CART_DISCOUNT);
    }

    public function remove(int $index): void
    {
        $cartItems = $this->session()->get(self::CART_KEY, []);

        if (isset($cartItems[$index])) {
            unset($cartItems[$index]);
            // Réindexe les éléments pour éviter les trous dans le tableau
            $this->session()->set(self::CART_KEY, array_values($cartItems));
        }
    }

    public function getDataForProforma(array $carts): array
    {
        $proforma = [];
        /**
         * @var int $cartKey
         * @var CartItem $cart
         */
        foreach ($carts as $cartKey => $cart) {
            $proforma['facture'][$cartKey] = [
                'id' => $cartKey,
                'quantite' => $cart->getQuantity(),
                'designation' => $cart->getDesignation(),
                'pu' => round($cart->getPrice()),
                'pt' => round($cart->getPrice() * $cart->getQuantity())
            ];
        }
        return $proforma;
    }
}