<?php

namespace App\Entity;

use App\Entity\Common\DatesTrait;
use App\Entity\Common\IdTrait;
use App\Enum\UserRoles;
use App\Repository\UsersRepository;
use DateTimeImmutable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\HttpFoundation\File\File;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Validator\Constraints as Assert;
use Vich\UploaderBundle\Mapping\Annotation as Vich;

#[ORM\Entity(repositoryClass: UsersRepository::class)]
#[ORM\HasLifecycleCallbacks]
#[ORM\UniqueConstraint(name: 'UNIQ_IDENTIFIER_EMAIL', fields: ['email'])]
#[Vich\Uploadable]
class Users implements UserInterface, PasswordAuthenticatedUserInterface
{
    use DatesTrait;
    use IdTrait;

    #[ORM\Column(type: 'string', length: 255, nullable: false)]
    #[Assert\NotBlank]
    private ?string $nomprenoms = null;

    #[ORM\Column(type: 'string', length: 255, nullable: true)]
    private ?string $contacts = null;

    #[ORM\Column(length: 180, unique: true)]
    #[Assert\NotBlank]
    #[Assert\Email]
    private ?string $email = null;

    #[ORM\Column(type: 'string', length: 255, nullable: true)]
    private ?string $username = null;

    #[ORM\Column(type: 'string', length: 255, nullable: true)]
    private ?string $adresse = null;

    #[ORM\Column(type: 'json')]
    private array $roles;

    #[ORM\Column]
    private ?string $password = null;

    #[ORM\Column(type: 'boolean', nullable: true, options: ["default" => false])]
    private ?bool $enabled = false;

    #[Vich\UploadableField(mapping: "users_photo", fileNameProperty: "filename")]
    #[Assert\Image(
        mimeTypes: ["image/png", "image/jpeg"],
        maxWidth: '196',
        maxHeight: '196',
        mimeTypesMessage: "Veuillez sélectionner une image valide"
    )]
    private ?File $imageFile = null;

    #[ORM\Column(type: 'string', length: 255, nullable: true)]
    private ?string $filename = null;

    #[ORM\OneToMany(targetEntity: Clientsentreprises::class, mappedBy: 'users')]
    private Collection $clientsentreprises;

    #[ORM\OneToMany(targetEntity: Proformaentreprises::class, mappedBy: 'users')]
    private Collection $proformaentreprises;

    #[ORM\OneToMany(targetEntity: Notes::class, mappedBy: 'users')]
    private Collection $notes;

    #[ORM\OneToMany(targetEntity: Mails::class, mappedBy: 'users')]
    private Collection $mails;

    #[ORM\OneToMany(targetEntity: Proformaparticuliers::class, mappedBy: 'users')]
    private Collection $proformaparticuliers;

    #[ORM\OneToMany(targetEntity: PaniersEntreprise::class, mappedBy: 'users')]
    private Collection $paniersEntreprise;

    #[ORM\OneToMany(targetEntity: PaniersParticulier::class, mappedBy: 'users')]
    private Collection $paniersParticulier;


    public function __construct()
    {
        $this->clientsentreprises = new ArrayCollection();
        $this->proformaentreprises = new ArrayCollection();
        $this->notes = new ArrayCollection();
        $this->mails = new ArrayCollection();
        $this->proformaparticuliers = new ArrayCollection();
        $this->paniersParticulier = new ArrayCollection();
        $this->paniersEntreprise = new ArrayCollection();
    }

    public function getNomprenoms(): ?string
    {
        return $this->nomprenoms;
    }

    public function setNomprenoms(string $nomprenoms): self
    {
        $this->nomprenoms = $nomprenoms;

        return $this;
    }

    public function getContacts(): ?string
    {
        return $this->contacts;
    }

    public function setContacts(string $contacts): self
    {
        $this->contacts = $contacts;

        return $this;
    }

    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(?string $email): Users
    {
        $this->email = $email;
        return $this;
    }

    public function getUsername(): ?string
    {
        return $this->username;
    }

    public function setUsername(?string $username): Users
    {
        $this->username = $username;
        return $this;
    }

    public function getAdresse(): ?string
    {
        return $this->adresse;
    }

    public function setAdresse(?string $adresse): self
    {
        $this->adresse = $adresse;

        return $this;
    }

    public function getRolesUsers(string $key): ?string
    {
        return UserRoles::getRolesUsers($key) ?? null;
    }

    public function isSuperAdmin(): bool
    {
        return UserRoles::isSuperAdmin($this->getRoles());
    }

    public function getRoles(): array
    {
        $roles = $this->roles;

        return array_unique($roles);
    }

    public function setRoles(array $roles): static
    {
        $this->roles = $roles;

        return $this;
    }

    public function isEntreprise(): bool
    {
        return UserRoles::isEntreprise($this->getRoles());
    }

    public function isParticulier(): bool
    {
        return UserRoles::isParticulier($this->getRoles());
    }

    public function getEnabled(): ?bool
    {
        return $this->enabled;
    }

    public function setEnabled(?bool $enabled): static
    {
        $this->enabled = $enabled;
        return $this;
    }

    public function getImageFile(): ?File
    {
        return $this->imageFile;
    }

    public function setImageFile(?File $imageFile): Users
    {
        $this->imageFile = $imageFile;
        if ($this->imageFile instanceof UploadedFile) {
            $this->updatedAt = new DateTimeImmutable('now');
        }
        return $this;
    }

    public function getFilename(): ?string
    {
        return $this->filename;
    }

    public function setFilename(?string $filename): self
    {
        $this->filename = $filename;

        return $this;
    }

    /**
     * @return Collection<int, Clientsentreprises>
     */
    public function getClientsentreprises(): Collection
    {
        return $this->clientsentreprises;
    }

    public function addClientsentreprise(Clientsentreprises $clientsentreprise): self
    {
        if (!$this->clientsentreprises->contains($clientsentreprise)) {
            $this->clientsentreprises[] = $clientsentreprise;
            $clientsentreprise->setUsers($this);
        }

        return $this;
    }

    public function removeClientsentreprise(Clientsentreprises $clientsentreprise): self
    {
        if ($this->clientsentreprises->contains($clientsentreprise)) {
            $this->clientsentreprises->removeElement($clientsentreprise);
            if ($clientsentreprise->getUsers() === $this) {
                $clientsentreprise->setUsers(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Proformaentreprises>
     */
    public function getProformaentreprises(): Collection
    {
        return $this->proformaentreprises;
    }

    public function addProformaentreprise(Proformaentreprises $proformaentreprise): self
    {
        if (!$this->proformaentreprises->contains($proformaentreprise)) {
            $this->proformaentreprises[] = $proformaentreprise;
            $proformaentreprise->setUsers($this);
        }

        return $this;
    }

    public function removeProformaentreprise(Proformaentreprises $proformaentreprise): self
    {
        if ($this->proformaentreprises->contains($proformaentreprise)) {
            $this->proformaentreprises->removeElement($proformaentreprise);
            // set the owning side to null (unless already changed)
            if ($proformaentreprise->getUsers() === $this) {
                $proformaentreprise->setUsers(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Notes>
     */
    public function getNotes(): Collection
    {
        return $this->notes;
    }

    public function addNote(Notes $note): self
    {
        if (!$this->notes->contains($note)) {
            $this->notes[] = $note;
            $note->setUsers($this);
        }

        return $this;
    }

    public function removeNote(Notes $note): self
    {
        if ($this->notes->contains($note)) {
            $this->notes->removeElement($note);
            if ($note->getUsers() === $this) {
                $note->setUsers(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Mails>
     */
    public function getMails(): Collection
    {
        return $this->mails;
    }

    public function addMail(Mails $mail): self
    {
        if (!$this->mails->contains($mail)) {
            $this->mails[] = $mail;
            $mail->setUsers($this);
        }

        return $this;
    }

    public function removeMail(Mails $mail): self
    {
        if ($this->mails->contains($mail)) {
            $this->mails->removeElement($mail);
            if ($mail->getUsers() === $this) {
                $mail->setUsers(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Proformaparticuliers>
     */
    public function getProformaparticuliers(): Collection
    {
        return $this->proformaparticuliers;
    }

    public function addProformaparticulier(Proformaparticuliers $proformaparticulier): self
    {
        if (!$this->proformaparticuliers->contains($proformaparticulier)) {
            $this->proformaparticuliers[] = $proformaparticulier;
            $proformaparticulier->setUsers($this);
        }

        return $this;
    }

    public function removeProformaparticulier(Proformaparticuliers $proformaparticulier): self
    {
        if ($this->proformaparticuliers->contains($proformaparticulier)) {
            $this->proformaparticuliers->removeElement($proformaparticulier);
            if ($proformaparticulier->getUsers() === $this) {
                $proformaparticulier->setUsers(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, PaniersEntreprise>
     */
    public function getPaniersEntreprise(): Collection
    {
        return $this->paniersEntreprise;
    }

    public function addPanierEntreprise(PaniersEntreprise $panier): self
    {
        if (!$this->paniersEntreprise->contains($panier)) {
            $this->paniersEntreprise[] = $panier;
            $panier->setUsers($this);
        }

        return $this;
    }

    public function removePanierEntreprise(PaniersEntreprise $panier): self
    {
        if ($this->paniersEntreprise->contains($panier)) {
            $this->paniersEntreprise->removeElement($panier);
            if ($panier->getUsers() === $this) {
                $panier->setUsers(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, PaniersParticulier>
     */
    public function getPaniersParticulier(): Collection
    {
        return $this->paniersParticulier;
    }

    public function addPanierParticulier(PaniersParticulier $panier): self
    {
        if (!$this->paniersParticulier->contains($panier)) {
            $this->paniersParticulier[] = $panier;
            $panier->setUsers($this);
        }

        return $this;
    }

    public function removePanierParticulier(PaniersParticulier $panier): self
    {
        if ($this->paniersParticulier->contains($panier)) {
            $this->paniersParticulier->removeElement($panier);
            if ($panier->getUsers() === $this) {
                $panier->setUsers(null);
            }
        }

        return $this;
    }

    /**
     * @see PasswordAuthenticatedUserInterface
     */
    public function getPassword(): string
    {
        return $this->password;
    }

    public function setPassword(string $password): static
    {
        $this->password = $password;

        return $this;
    }

    public function eraseCredentials(): void
    {
        // TODO: Implement eraseCredentials() method.
    }

    public function getUserIdentifier(): string
    {
        return (string)$this->email;
    }
}
