<?php

namespace App\Controller\Admin;

use App\Entity\Users;
use App\Form\UsersType;
use App\Managers\Managers;
use App\Repository\UsersRepository;
use App\Services\Helpers\FlashServiceInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/dashboard/utilisateurs')]
#[IsGranted(new Expression('is_granted("ROLE_SUPER_ADMIN")'))]
class AdminUsersController extends AbstractController
{
    public function __construct(
        private readonly Managers                    $manager,
        private readonly FlashServiceInterface       $flash,
        private readonly UsersRepository             $repository,
        private readonly UserPasswordHasherInterface $encoder,
    )
    {
    }

    #[Route('/', name: "admin.utilisateurs.index", methods: ['GET'])]
    public function index(): Response
    {
        return $this->render('admin/utilisateurs/index.html.twig', [
            'users' => $this->repository->findAll(),
            'title' => 'Liste des utilisateurs',
            'libelle_ajouter' => 'Nouvel utilisateur',
        ]);
    }

    #[Route('/new', name: "admin.utilisateurs.new", methods: ['GET', 'POST'])]
    public function new(Request $request): Response
    {
        $user = new Users();
        $form = $this->createForm(UsersType::class, $user);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $user->setUsername(strtolower($form->get('username')->getData()));
            $user->setEmail(strtolower($form->get('email')->getData()));
            $user->setEnabled(true);
            $user->setPassword($this->encoder->hashPassword($user, $form->get('password')->getData()));
            $this->manager->setEntity($user, 'new');
            $this->flash->success("Enrégistrement effectué avec succès.");
            return $this->redirectToRoute('admin.utilisateurs.new');
        }

        return $this->render('admin/utilisateurs/new.html.twig', [
            'user' => $user,
            'form' => $form->createView(),
            'title' => 'Ajouter un nouvel utilisateur',
            'libelle_liste' => 'Liste des utilisateurs',
            'libelle_ajouter' => 'Nouvel utilisateur',
        ]);
    }

    #[Route('/show/{id}', name: 'admin.utilisateurs.show', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function show(Users $user): Response
    {
        return $this->render('admin/utilisateurs/show.html.twig', [
            'user' => $user,
            'title' => 'Détails utilisateur',
            'libelle_liste' => 'Liste des utilisateurs',
            'libelle_ajouter' => 'Nouvel utilisateur',
        ]);
    }

    #[Route('/{id}/edit', name: 'admin.utilisateurs.edit', requirements: ['id' => Requirement::DIGITS], methods: ['GET', 'POST'])]
    public function edit(Request $request, Users $user): Response
    {
        $form = $this->createForm(UsersType::class, $user);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $user->setUsername(strtolower($form->get('username')->getData()));
            $user->setEmail(strtolower($form->get('email')->getData()));
            $user->setEnabled(true);
            $this->manager->setEntity($user, 'edit');
            $this->flash->success("Modification effectuée avec succès.");
            return $this->redirectToRoute('admin.utilisateurs.index', [
                'id' => $user->getId(),
            ]);
        }

        return $this->render('admin/utilisateurs/edit.html.twig', [
            'user' => $user,
            'form' => $form->createView(),
            'title' => 'Editer un utilisateur',
            'libelle_ajouter' => 'Nouvel utilisateur',
            'libelle_liste' => 'Liste des utilisateurs',
        ]);
    }

    #[Route('/{id}', name: 'admin.utilisateurs.delete', requirements: ['id' => Requirement::DIGITS], methods: ['DELETE'])]
    public function delete(Request $request, Users $user): Response
    {
        if ($this->isCsrfTokenValid('delete' . $user->getId(), $request->request->get('_token'))) {
            $this->manager->setEntity($user, 'delete');
            $this->flash->success("Suppression effectuée avec succès.");
        }

        return $this->redirectToRoute('admin.utilisateurs.index');
    }

    #[Route('/locked/{id}', name: 'utilisateurs.locked', requirements: ['id' => Requirement::DIGITS], methods: ['GET'])]
    public function lockedUsers(Request $request, Users $user): Response
    {
        $active = $request->query->get('active');
        if ($active === "on") {
            $user->setEnabled(true);
            $this->manager->setEntity($user, 'edit');
            $this->flash->success("Utilisateur débloqué.");
        } else {
            $user->setEnabled(false);
            $this->manager->setEntity($user, 'edit');
            $this->flash->success("Utilisateur bloqué.");
        }
        return $this->redirectToRoute('admin.utilisateurs.index');
    }
}
